
package kom.itbeankit.common;


/**
 * This class defines a filename in the following format:
 * "beanNumber-iterationNumber-dataType-name.suffix".
 * beanNumber: continues number of the ItBean given from the framework
 * iterationNumber: continues number of iteration, e.g. frame number of frame sequence
 * datatype: datatype number as specified for Iteachs datatypes
 * name: specific name of the file or file sequence
 * suffix: usually used filename suffix (or extension) for file type
 * Examples of accepting filename formats:
 * name                                   name.ext
 * name005.ext                            005-name.ext
 * 4530-name005.ext                       005-4530-name.ext
 * 14-4530-name005.ext                    14-005-4530-name.ext
 * (beanNumber=14, iterationNumber=5, datatype = 4530, name = "name", suffix = "ext")
 *
 * @author: Christian C. Hoffmann
 * @version: 1.01 08/04/98
 */
public class ItFilename {
	private int beanNumber;
	private int datatype;
	private int iterationNumber;
	private String name;
	private String suffix;
	private boolean iterationNumberIsInName = false;



/**
 * This contructor creates an ItFilename
 * containing a default name.
 */
public ItFilename() {
	this("default000.xxx");
}
/**
 * This contructor creates a standardized ItFilename
 * by analyzing the composite filename.
 * @param filename java.lang.String
 */
public ItFilename(String filename) {
	this.suffix = extractExtension(filename);
	this.iterationNumber = extractIterationNumber(filename);
	this.name = extractName(filename);
	this.datatype = extractDatatype(filename);
	this.beanNumber = extractBeanNumber(filename);
}
/**
 * This method counts the chars c in the string s.
 * @return int
 * @param s java.lang.String
 * @param c char
 */
public static int countOf(String s, char c) {
	int count = 0;
	for (int i=0; i<s.length(); i++) 
		if (s.charAt(i) == c) count++;
	return count;
}
/**
 * This method extracts the BeanNumber out of the filename.
 * @return int -1, if bean number not found
 * @param filename java.lang.String
 */
private int extractBeanNumber(String filename) {
	switch (countOf(filename,'-')) {
	case 0:  return -1;
	case 1:  return -1;
	case 2:  if (this.iterationNumberIsInName) return numberAt(filename,1);
				  else return -1;
	case 3:  return numberAt(filename,1);
	}	
	return -1;
}
/**
 * This method extracts the datatype number out of the filename.
 * @return int -1 if no datatype number found
 * @param filename java.lang.String
 */
private int extractDatatype(String filename) {
	switch (countOf(filename,'-')) {
	case 0:  return -1;
	case 1:  if (this.iterationNumberIsInName) return numberAt(filename,1);
				  else return -1;
	case 2:  return numberAt(filename,2);
	case 3:  return numberAt(filename,3);
	}	
	return -1;
}
/**
 * This method extracts the extension out of a filename String.
 * @param filename java.lang.String
 */
private String extractExtension(String filename) {
	int pos = filename.lastIndexOf(".");
	if (pos>=0) {
		return filename.substring(pos+1);
	}	
	return "";
}
/**
 * This method extracts the IterationNumber out of a filename.
 * @param filename java.lang.String
 */
private int extractIterationNumber(String filename) {
	char[] c = new char[filename.length()];
	filename.getChars(0,filename.length()-1,c,0);
	int pos = c.length-1;
	while (!java.lang.Character.isDigit(c[pos]) && (pos>0)) pos--;
	if (c[pos+1] == '.') {        // iteration number is part of name
		this.iterationNumberIsInName = true;
		return numberAt(filename,countOf(filename,'-')+1);
	}	
	else {                        // iteration number is not part of name
		int j = countOf(filename,'-');
		switch (j) {
			case 1: return numberAt(filename,1);
			case 2: return numberAt(filename,1);
			case 3: return numberAt(filename,2);
		}
	}
	return -1;
}			
/**
 * This method extracts the name out of the filename.
 * @return java.lang.String the name 
 * @param filename java.lang.String the filename
 */
private String extractName(String filename) {
	int end = filename.length()-1;
	if (this.suffix != "")
		while ((end > 0) && (filename.charAt(end)!='.')) end--;
		while ((end > 0) && (java.lang.Character.isDigit(filename.charAt(end-1)))) end--;
	int begin = end;
	while ((begin > 0) && (filename.charAt(begin-1) != '-')) begin--; 
	return filename.substring(begin,end);
}
/**
 * This method returns the BeanNumber property.
 * @return int
 */
public int getBeanNumber() {
	return this.beanNumber;
}
/**
 * This method returns the datatype property.
 * @return int
 */
public int getDatatype() {
	return this.datatype;
}
/**
 * This method returns a composite filename.
 * @return String 
 */
public String getFilename() {
	String filename = "";
	if ((this.beanNumber>=0) && (this.iterationNumber>=0) && (this.datatype>=0)) 
		filename = kom.itbeankit.util.Convert.int2string(this.beanNumber,2) + "-"
		+ kom.itbeankit.util.Convert.int2string(this.iterationNumber,3) + "-"
		+ kom.itbeankit.util.Convert.int2string(this.datatype,4) + "-" + this.name + "." + this.suffix;
	if ((this.beanNumber<0) && (this.iterationNumber>=0) && (this.datatype>=0)) 
		filename = kom.itbeankit.util.Convert.int2string(this.datatype,4) + "-" + this.name 
		+ kom.itbeankit.util.Convert.int2string(this.iterationNumber,3) + "." + this.suffix;
	if ((this.beanNumber<0) && (this.iterationNumber>=0) && (this.datatype<0)) 
		filename = this.name 
		+ kom.itbeankit.util.Convert.int2string(this.iterationNumber,3) + "." + this.suffix;
	if ((this.beanNumber<0) && (this.iterationNumber<0) && (this.datatype<0)) 
		filename = this.name + "." + this.suffix;
	return filename;
}
/**
 * This method returns the iteration number property.
 * @return int
 */
public int getIterationNumber() {
	return this.iterationNumber;
}
/**
 * This method returns the IterationNumberIsInName property.
 * @return boolean
 */
public boolean getIterationNumberIsInName() {
	return this.iterationNumberIsInName;
}
/**
 * This method returns the name property.
 * @return java.lang.String
 */
public String getName() {
	return this.name;
}
/**
 * This method returns the suffix property.
 * @return java.lang.String
 */
public String getSuffix() {
	return this.suffix;
}
/**
 * This method returns the posth number in String s. 
 * @return int the number no. pos, -1 if number does not exist
 * @param s java.lang.String
 * @param pos int
 */
public static int numberAt(String s, int pos) {
	char[] c = new char[s.length()];
	s.getChars(0,s.length()-1,c,0);
	int i = 0;
	int j = 0;
	int begin = 0;
	int end = 0;
	do {
		while (!java.lang.Character.isDigit(c[i]) && (i<s.length())) i++;
		begin = i;
		while (java.lang.Character.isDigit(c[i]) && (i<s.length())) i++;
		end = i;
		j++;
	} 
	while ((j < pos) && (i<s.length()));
	if (i<s.length()) return java.lang.Integer.parseInt(s.substring(begin,end));
	return -1;
}
/**
 * This method sets the bean number property.
 * @param beanNo int
 */
public void setBeanNumber(int beanNo) {
	this.beanNumber = beanNo;
}
/**
 * This method sets the datatype property.
 * @param dt int
 */
public void setDatatype(int dt) {
	this.datatype = dt;
}
/**
 * This method sets the iteration number property.
 * @param itno int
 */
public void setIterationNumber(int itno) {
	this.iterationNumber = itno;
}
/**
 * This method sets the IterationNumberIsInName flag.
 * @param flag the new flag value
 */
public void setIterationNumberIsInName(boolean flag) {
	this.iterationNumberIsInName = flag;
}
/**
 * This method sets the name property.
 * @param name java.lang.String
 */
public void setName(String name) {
	this.name = name;
}
/**
 * This method sets the suffix property.
 * @param suffix java.lang.String
 */
public void setSuffix(String suffix) {
	this.suffix = suffix;
}
}